
import os
import json
import re
import time
import tempfile
import sqlite3
import urllib.parse
from pathlib import Path
from datetime import datetime, timezone, timedelta

import requests
from flask import Flask, request, abort, jsonify
from dotenv import load_dotenv
from PIL import Image

load_dotenv()

BOT_TOKEN = os.environ.get("BOT_TOKEN", "")
WEBHOOK_SECRET = os.environ.get("WEBHOOK_SECRET", "")
SQLITE_PATH = os.environ.get("SQLITE_PATH", "bot.db")

BASE_URL = os.environ.get("BASE_URL", "").rstrip("/")
WEBHOOK_PATH = os.environ.get("WEBHOOK_PATH", "/tg/webhook")

TG_API = f"https://api.telegram.org/bot{BOT_TOKEN}"
TG_FILE_API = f"https://api.telegram.org/file/bot{BOT_TOKEN}"

app = Flask(__name__)

# ---------- DB ----------
def db():
    conn = sqlite3.connect(SQLITE_PATH)
    conn.row_factory = sqlite3.Row
    return conn

def init_db():
    conn = db()
    cur = conn.cursor()
    cur.executescript("""
    CREATE TABLE IF NOT EXISTS users (
      user_id INTEGER PRIMARY KEY,
      first_name TEXT,
      username TEXT,
      plan TEXT DEFAULT 'normal',
      is_blocked INTEGER DEFAULT 0,
      created_at TEXT,
      last_active TEXT,
      last_bot_msg_id INTEGER,
      state TEXT,
      state_json TEXT
    );

    CREATE TABLE IF NOT EXISTS settings (
      k TEXT PRIMARY KEY,
      v TEXT
    );

    CREATE TABLE IF NOT EXISTS join_channels (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      chat_ref TEXT NOT NULL,           -- @username or -100...
      title TEXT,
      invite_link TEXT,                -- optional, for private
      created_at TEXT
    );

    CREATE TABLE IF NOT EXISTS upgrade_requests (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      user_id INTEGER NOT NULL,
      requested_plan TEXT NOT NULL,
      status TEXT DEFAULT 'pending',    -- pending/approved/rejected
      created_at TEXT,
      decided_at TEXT,
      decided_by INTEGER
    );

    CREATE TABLE IF NOT EXISTS admin_logs (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      admin_id INTEGER,
      action TEXT,
      detail TEXT,
      created_at TEXT
    );

    CREATE TABLE IF NOT EXISTS daily_usage (
      user_id INTEGER,
      yyyymmdd TEXT,
      files_count INTEGER DEFAULT 0,
      PRIMARY KEY (user_id, yyyymmdd)
    );
    """)
    conn.commit()
    conn.close()

def get_setting(key, default=None):
    conn = db()
    cur = conn.cursor()
    cur.execute("SELECT v FROM settings WHERE k=?", (key,))
    row = cur.fetchone()
    conn.close()
    if not row:
        return default
    try:
        return json.loads(row["v"])
    except Exception:
        return row["v"]

def set_setting(key, value):
    conn = db()
    cur = conn.cursor()
    cur.execute(
        "INSERT INTO settings(k,v) VALUES(?,?) ON CONFLICT(k) DO UPDATE SET v=excluded.v",
        (key, json.dumps(value, ensure_ascii=False)),
    )
    conn.commit()
    conn.close()

def now_iso():
    return datetime.now(timezone.utc).isoformat()

def load_config():
    cfg_path = Path(__file__).with_name("config.json")
    if cfg_path.exists():
        with open(cfg_path, "r", encoding="utf-8") as f:
            return json.load(f)
    return {}

CONFIG = load_config()

def ensure_defaults():
    if get_setting("min_size_bytes") is None:
        set_setting("min_size_bytes", int(CONFIG.get("min_size_bytes", 1_048_576)))
    if get_setting("max_size_bytes") is None:
        set_setting("max_size_bytes", int(CONFIG.get("max_size_bytes", 3_221_225_472)))
    if get_setting("plan_limits") is None:
        set_setting(
            "plan_limits",
            CONFIG.get(
                "plan_limits",
                {
                    "normal": {"max_size_bytes": 50 * 1024 * 1024, "daily_files": 5},
                    "pro": {"max_size_bytes": 500 * 1024 * 1024, "daily_files": 50},
                    "admin": {"max_size_bytes": 3 * 1024 * 1024 * 1024, "daily_files": 1_000_000},
                },
            ),
        )
    if get_setting("join_required") is None:
        set_setting("join_required", True)
    if get_setting("msg_start") is None:
        set_setting(
            "msg_start",
            "سلام! 👋\nاین ربات اسم فایل‌هات رو عوض می‌کنه و دوباره تحویلت می‌ده.\nاز منوی زیر انتخاب کن 👇",
        )
    if get_setting("msg_help") is None:
        set_setting(
            "msg_help",
            "▫️ فایل رو بفرست\n▫️ اسم جدید رو وارد کن\n▫️ فایل با نام جدید تحویل بگیر\n\nنکته: برای عکس‌ها خروجی به صورت فایل (Document) ارسال می‌شود.",
        )
    if get_setting("msg_force_join") is None:
        set_setting(
            "msg_force_join",
            "برای استفاده از ربات باید عضو کانال‌های زیر بشی 👇\nبعدش روی «✅ عضو شدم» بزن",
        )
    if get_setting("msg_cancelled") is None:
        set_setting("msg_cancelled", "عملیات لغو شد ❌\nاز منوی زیر انتخاب کن 👇")

# ---------- Telegram helpers ----------
def tg(method, payload=None, files=None, timeout=60):
    if not BOT_TOKEN:
        raise RuntimeError("BOT_TOKEN is not set")
    url = f"{TG_API}/{method}"
    r = requests.post(url, data=payload or {}, files=files, timeout=timeout)
    if not r.ok:
        return {"ok": False, "error": r.text, "status": r.status_code}
    return r.json()

def tg_get(method, params=None, timeout=60):
    url = f"{TG_API}/{method}"
    r = requests.get(url, params=params or {}, timeout=timeout)
    if not r.ok:
        return {"ok": False, "error": r.text, "status": r.status_code}
    return r.json()

def safe_delete(chat_id, message_id):
    try:
        tg("deleteMessage", {"chat_id": chat_id, "message_id": message_id})
    except Exception:
        pass

def upsert_user(u):
    conn = db()
    cur = conn.cursor()
    cur.execute("SELECT user_id, plan FROM users WHERE user_id=?", (u["id"],))
    row = cur.fetchone()

    admins = set(CONFIG.get("admins", []))
    plan = "admin" if u["id"] in admins else ("admin" if row and row["plan"] == "admin" else "normal")

    if row:
        cur.execute(
            "UPDATE users SET first_name=?, username=?, last_active=?, plan=? WHERE user_id=?",
            (u.get("first_name"), u.get("username"), now_iso(), plan, u["id"]),
        )
    else:
        cur.execute(
            "INSERT INTO users(user_id, first_name, username, plan, created_at, last_active) VALUES(?,?,?,?,?,?)",
            (u["id"], u.get("first_name"), u.get("username"), plan, now_iso(), now_iso()),
        )
    conn.commit()
    conn.close()

def get_user(user_id):
    conn = db()
    cur = conn.cursor()
    cur.execute("SELECT * FROM users WHERE user_id=?", (user_id,))
    row = cur.fetchone()
    conn.close()
    return dict(row) if row else None

def set_user_state(user_id, state, data=None):
    conn = db()
    cur = conn.cursor()
    cur.execute(
        "UPDATE users SET state=?, state_json=? WHERE user_id=?",
        (state, json.dumps(data or {}, ensure_ascii=False) if state else None, user_id),
    )
    conn.commit()
    conn.close()

def set_last_bot_msg(user_id, msg_id):
    conn = db()
    cur = conn.cursor()
    cur.execute("UPDATE users SET last_bot_msg_id=? WHERE user_id=?", (msg_id, user_id))
    conn.commit()
    conn.close()

def get_state(user_id):
    u = get_user(user_id)
    if not u:
        return None, {}
    st = u.get("state")
    try:
        data = json.loads(u.get("state_json") or "{}")
    except Exception:
        data = {}
    return st, data

def log_admin(admin_id, action, detail=""):
    conn = db()
    cur = conn.cursor()
    cur.execute("INSERT INTO admin_logs(admin_id, action, detail, created_at) VALUES(?,?,?,?)", (admin_id, action, detail, now_iso()))
    conn.commit()
    conn.close()

# ---------- Limits / plans ----------
def bytes_human(n):
    units = ["B", "KB", "MB", "GB", "TB"]
    x = float(n)
    i = 0
    while x >= 1024 and i < len(units) - 1:
        x /= 1024
        i += 1
    if i == 0:
        return f"{int(x)} {units[i]}"
    return f"{x:.2f} {units[i]}"

def parse_size(s):
    s = (s or "").strip().upper().replace(" ", "")
    m = re.fullmatch(r"(\\d+(?:\\.\\d+)?)(B|KB|MB|GB)?", s)
    if not m:
        return None
    val = float(m.group(1))
    unit = m.group(2) or "B"
    mult = {"B": 1, "KB": 1024, "MB": 1024 ** 2, "GB": 1024 ** 3}.get(unit, 1)
    return int(val * mult)

def get_user_limits(user):
    plan_limits = get_setting("plan_limits", {})
    plan = user.get("plan", "normal")
    pl = plan_limits.get(plan, plan_limits.get("normal", {"max_size_bytes": 50 * 1024 * 1024, "daily_files": 5}))
    global_max = int(get_setting("max_size_bytes", 3_221_225_472))
    global_min = int(get_setting("min_size_bytes", 1_048_576))
    max_size = global_max if plan == "admin" else min(int(pl.get("max_size_bytes", global_max)), global_max)
    daily_files = 1_000_000 if plan == "admin" else int(pl.get("daily_files", 5))
    return {"max_size": max_size, "min_size": global_min, "daily_files": daily_files}

def bump_daily_usage(user_id):
    today = datetime.now(timezone.utc).strftime("%Y%m%d")
    conn = db()
    cur = conn.cursor()
    cur.execute("SELECT files_count FROM daily_usage WHERE user_id=? AND yyyymmdd=?", (user_id, today))
    row = cur.fetchone()
    if row:
        newv = int(row["files_count"]) + 1
        cur.execute("UPDATE daily_usage SET files_count=? WHERE user_id=? AND yyyymmdd=?", (newv, user_id, today))
    else:
        newv = 1
        cur.execute("INSERT INTO daily_usage(user_id, yyyymmdd, files_count) VALUES(?,?,?)", (user_id, today, newv))
    conn.commit()
    conn.close()
    return newv

def get_daily_usage(user_id):
    today = datetime.now(timezone.utc).strftime("%Y%m%d")
    conn = db()
    cur = conn.cursor()
    cur.execute("SELECT files_count FROM daily_usage WHERE user_id=? AND yyyymmdd=?", (user_id, today))
    row = cur.fetchone()
    conn.close()
    return int(row["files_count"]) if row else 0

# ---------- UI builders ----------
def ik(button_rows):
    return json.dumps({"inline_keyboard": button_rows}, ensure_ascii=False)

def main_menu_kb(is_admin):
    rows = [
        [{"text": "📁 تغییر نام فایل", "callback_data": "U:RENAME_FILE"}],
        [{"text": "🔗 تغییر نام از لینک", "callback_data": "U:RENAME_LINK"}],
        [{"text": "👤 پروفایل من", "callback_data": "U:PROFILE"}],
        [{"text": "ℹ️ راهنما", "callback_data": "U:HELP"}],
    ]
    if is_admin:
        rows.append([{"text": "👑 پنل ادمین", "callback_data": "A:MENU"}])
    rows.append([{"text": "❌ لغو", "callback_data": "U:CANCEL"}])
    return ik(rows)

def cancel_menu_kb(is_admin):
    rows = [
        [{"text": "📁 تغییر نام فایل", "callback_data": "U:RENAME_FILE"}],
        [{"text": "🔗 تغییر نام از لینک", "callback_data": "U:RENAME_LINK"}],
        [{"text": "👤 پروفایل من", "callback_data": "U:PROFILE"}],
        [{"text": "ℹ️ راهنما", "callback_data": "U:HELP"}],
    ]
    if is_admin:
        rows.append([{"text": "👑 پنل ادمین", "callback_data": "A:MENU"}])
    return ik(rows)

def join_gate_kb():
    chans = list_join_channels()
    rows = []
    for ch in chans:
        link = ch.get("invite_link")
        ref = ch.get("chat_ref")
        if ref.startswith("@"):
            link = link or f"https://t.me/{ref[1:]}"
        if link:
            rows.append([{"text": f"🔗 {ch.get('title') or ref}", "url": link}])
    rows.append([{"text": "✅ عضو شدم", "callback_data": "J:CHECK"}])
    rows.append([{"text": "❌ لغو", "callback_data": "U:CANCEL"}])
    return ik(rows)

def file_action_kb():
    return ik([
        [{"text": "✏️ تغییر نام", "callback_data": "U:SET_NAME"}],
        [{"text": "🔄 تغییر فرمت", "callback_data": "U:SET_FMT"}],
        [{"text": "❌ لغو", "callback_data": "U:CANCEL"}],
    ])

def confirm_kb():
    return ik([
        [{"text": "✅ تأیید و ارسال", "callback_data": "U:SEND"}],
        [{"text": "✏️ ویرایش نام", "callback_data": "U:SET_NAME"}],
        [{"text": "🔄 تغییر فرمت", "callback_data": "U:SET_FMT"}],
        [{"text": "❌ لغو", "callback_data": "U:CANCEL"}],
    ])

def format_kb_for(kind):
    if kind == "image":
        return ik([
            [{"text": "🔄 تبدیل به JPG", "callback_data": "F:JPG"}],
            [{"text": "🔄 تبدیل به PNG", "callback_data": "F:PNG"}],
            [{"text": "🔄 تبدیل به WEBP", "callback_data": "F:WEBP"}],
            [{"text": "🔙 برگشت", "callback_data": "U:BACK_FILE"}],
            [{"text": "❌ لغو", "callback_data": "U:CANCEL"}],
        ])
    return ik([
        [{"text": "(فعلاً برای این نوع فایل فعال نیست)", "callback_data": "NOOP"}],
        [{"text": "🔙 برگشت", "callback_data": "U:BACK_FILE"}],
        [{"text": "❌ لغو", "callback_data": "U:CANCEL"}],
    ])

# ---------- Force join ----------
def list_join_channels():
    conn = db()
    cur = conn.cursor()
    cur.execute("SELECT chat_ref, title, invite_link FROM join_channels ORDER BY id ASC")
    rows = cur.fetchall()
    conn.close()
    return [dict(r) for r in rows]

def check_user_joined_all(user_id):
    if not get_setting("join_required", True):
        return True, ""
    channels = list_join_channels()
    if not channels:
        return True, ""
    for ch in channels:
        ref = ch["chat_ref"]
        res = tg_get("getChatMember", {"chat_id": ref, "user_id": user_id}, timeout=20)
        if not res.get("ok"):
            return False, f"❗️امکان بررسی عضویت در {ref} وجود ندارد. (ربات باید ادمین باشد)"
        status = res["result"]["status"]
        if status in ("left", "kicked"):
            return False, ""
    return True, ""

# ---------- File helpers ----------
def sanitize_filename(name):
    name = (name or "").strip()
    name = name.replace("\\\\", "_").replace("/", "_")
    name = re.sub(r"[\\x00-\\x1f<>:\\\\\"|?*]+", "_", name)
    name = re.sub(r"\\s+", " ", name)
    return name[:180] if len(name) > 180 else name

def detect_kind(mime, filename):
    mime = (mime or "").lower()
    ext = (Path(filename).suffix or "").lower()
    if mime.startswith("image/") or ext in (".jpg", ".jpeg", ".png", ".webp", ".bmp", ".tiff", ".gif"):
        return "image"
    return "other"

def get_file_info_from_message(msg):
    if msg.get("document"):
        d = msg["document"]
        return {"src": "tg", "file_id": d["file_id"], "filename": d.get("file_name") or "file", "mime": d.get("mime_type") or "", "size": d.get("file_size", 0) or 0}
    if msg.get("photo"):
        p = msg["photo"][-1]
        return {"src": "tg", "file_id": p["file_id"], "filename": "photo.jpg", "mime": "image/jpeg", "size": p.get("file_size", 0) or 0}
    if msg.get("video"):
        v = msg["video"]
        return {"src": "tg", "file_id": v["file_id"], "filename": v.get("file_name") or "video.mp4", "mime": v.get("mime_type") or "", "size": v.get("file_size", 0) or 0}
    if msg.get("audio"):
        a = msg["audio"]
        return {"src": "tg", "file_id": a["file_id"], "filename": a.get("file_name") or "audio.mp3", "mime": a.get("mime_type") or "", "size": a.get("file_size", 0) or 0}
    return None

def tg_download(file_id, dest_path):
    res = tg_get("getFile", {"file_id": file_id}, timeout=60)
    if not res.get("ok"):
        raise RuntimeError("getFile failed: " + str(res))
    file_path = res["result"]["file_path"]
    url = f"{TG_FILE_API}/{file_path}"
    with requests.get(url, stream=True, timeout=180) as r:
        r.raise_for_status()
        with open(dest_path, "wb") as f:
            for chunk in r.iter_content(chunk_size=1024 * 512):
                if chunk:
                    f.write(chunk)

def download_url_to(url, dest_path, max_bytes):
    parsed = urllib.parse.urlparse(url)
    if parsed.scheme not in ("http", "https"):
        raise ValueError("URL scheme must be http/https")
    host = parsed.hostname or ""
    blocked_prefixes = ("127.", "0.", "169.254.", "10.", "192.168.", "172.16.", "172.17.", "172.18.", "172.19.", "::1")
    if host in ("localhost",) or host.startswith(blocked_prefixes):
        raise ValueError("Blocked host")
    # HEAD size check (best-effort)
    try:
        h = requests.head(url, allow_redirects=True, timeout=20)
        if "Content-Length" in h.headers:
            size = int(h.headers["Content-Length"])
            if size > max_bytes:
                raise ValueError("File too large")
    except Exception:
        pass

    downloaded = 0
    with requests.get(url, stream=True, allow_redirects=True, timeout=180) as r:
        r.raise_for_status()
        with open(dest_path, "wb") as f:
            for chunk in r.iter_content(chunk_size=1024 * 512):
                if not chunk:
                    continue
                downloaded += len(chunk)
                if downloaded > max_bytes:
                    raise ValueError("File too large")
                f.write(chunk)
    return downloaded

def guess_filename_from_url(url):
    parsed = urllib.parse.urlparse(url)
    name = Path(parsed.path).name
    return name or "downloaded_file"

# ---------- Send message with cleanup ----------
def send_clean(chat_id, user_id, text, reply_markup=None):
    u = get_user(user_id)
    if u and u.get("last_bot_msg_id"):
        safe_delete(chat_id, int(u["last_bot_msg_id"]))
    payload = {"chat_id": chat_id, "text": text, "parse_mode": "HTML"}
    if reply_markup:
        payload["reply_markup"] = reply_markup
    res = tg("sendMessage", payload)
    if res.get("ok"):
        set_last_bot_msg(user_id, res["result"]["message_id"])
    return res

# ---------- User menus ----------
def show_join_gate(chat_id, user_id, extra_note=""):
    text = get_setting("msg_force_join", "")
    if extra_note:
        text = extra_note + "\\n\\n" + text
    send_clean(chat_id, user_id, text, join_gate_kb())
    set_user_state(user_id, "JOIN_GATE", {})

def show_main(chat_id, user_id):
    u = get_user(user_id)
    is_admin = (u.get("plan") == "admin")
    send_clean(chat_id, user_id, get_setting("msg_start", ""), main_menu_kb(is_admin))
    set_user_state(user_id, None, None)

def show_help(chat_id, user_id):
    u = get_user(user_id)
    is_admin = (u.get("plan") == "admin")
    kb = ik([
        [{"text": "🏠 منوی اصلی", "callback_data": "U:HOME"}],
        [{"text": "❌ لغو", "callback_data": "U:CANCEL"}],
    ])
    send_clean(chat_id, user_id, get_setting("msg_help", ""), kb)
    set_user_state(user_id, None, None)

def show_profile(chat_id, user_id):
    u = get_user(user_id)
    is_admin = (u.get("plan") == "admin")
    lim = get_user_limits(u)
    used = get_daily_usage(user_id)
    ok, _ = check_user_joined_all(user_id)
    join_status = "✅ عضو هستید" if ok else "❌ عضو نیستید"
    plan_map = {"normal": "عادی", "pro": "ویژه", "admin": "ادمین"}
    text = (
        f"👤 <b>پروفایل شما</b>\\n\\n"
        f"👤 نام: <code>{(u.get('first_name') or '—')}</code>\\n"
        f"🆔 شناسه: <code>{user_id}</code>\\n"
        f"⭐ پلن: <b>{plan_map.get(u.get('plan','normal'), u.get('plan'))}</b>\\n"
        f"📦 سقف حجم مجاز: <b>{bytes_human(lim['max_size'])}</b>\\n"
        f"📈 استفاده امروز: <b>{used}</b> / <b>{lim['daily_files'] if lim['daily_files'] < 1_000_000 else '∞'}</b>\\n"
        f"🔒 عضویت کانال‌ها: <b>{join_status}</b>\\n"
    )
    rows = [
        [{"text": "🏷 پلن‌ها و ارتقا", "callback_data": "U:PLANS"}],
        [{"text": "🏠 منوی اصلی", "callback_data": "U:HOME"}],
        [{"text": "❌ لغو", "callback_data": "U:CANCEL"}],
    ]
    if is_admin:
        rows.insert(1, [{"text": "👑 پنل ادمین", "callback_data": "A:MENU"}])
    send_clean(chat_id, user_id, text, ik(rows))

def show_plans(chat_id, user_id):
    u = get_user(user_id)
    cur_plan = u.get("plan", "normal")
    plan_limits = get_setting("plan_limits", {})

    def plan_line(pkey, title):
        pl = plan_limits.get(pkey, {})
        mx = int(pl.get("max_size_bytes", 0) or 0)
        df = pl.get("daily_files", "—")
        return f"{title}\\n• سقف حجم: {bytes_human(mx)}\\n• سقف روزانه: {df}"

    text = (
        "🏷 <b>پلن‌ها</b>\\n\\n"
        f"🆓 <b>عادی</b>\\n{plan_line('normal','')}\\n\\n"
        f"⭐ <b>ویژه</b>\\n{plan_line('pro','')}\\n\\n"
        "👑 <b>ادمین</b>\\n(فقط برای مدیریت)"
    )
    rows = []
    if cur_plan == "normal":
        rows.append([{"text": "🔼 درخواست ارتقا به ویژه", "callback_data": "U:REQ_UP:pro"}])
    elif cur_plan == "pro":
        rows.append([{"text": "✅ پلن فعلی: ویژه", "callback_data": "NOOP"}])
    else:
        rows.append([{"text": "✅ شما ادمین هستید", "callback_data": "NOOP"}])
    rows += [
        [{"text": "👤 پروفایل من", "callback_data": "U:PROFILE"}],
        [{"text": "🏠 منوی اصلی", "callback_data": "U:HOME"}],
        [{"text": "❌ لغو", "callback_data": "U:CANCEL"}],
    ]
    send_clean(chat_id, user_id, text, ik(rows))

def create_upgrade_request(user_id, requested_plan):
    conn = db()
    cur = conn.cursor()
    cur.execute("SELECT id FROM upgrade_requests WHERE user_id=? AND status='pending'", (user_id,))
    if cur.fetchone():
        conn.close()
        return False
    cur.execute("INSERT INTO upgrade_requests(user_id, requested_plan, status, created_at) VALUES(?,?,?,?)", (user_id, requested_plan, "pending", now_iso()))
    conn.commit()
    conn.close()
    return True

def notify_admins(text):
    for aid in CONFIG.get("admins", []):
        try:
            tg("sendMessage", {"chat_id": aid, "text": text, "parse_mode": "HTML"})
        except Exception:
            pass

# ---------- Admin menus ----------
def admin_menu_kb():
    return ik([
        [{"text": "📊 وضعیت ربات", "callback_data": "A:STATUS"}],
        [{"text": "📦 تنظیم سقف حجم فایل", "callback_data": "A:MAXSIZE"}],
        [{"text": "📝 پیام‌ها (ارسال همگانی)", "callback_data": "A:BCAST"}],
        [{"text": "🔒 جوین اجباری", "callback_data": "A:JOIN"}],
        [{"text": "📩 درخواست‌های ارتقا", "callback_data": "A:UPREQ"}],
        [{"text": "🏠 منوی اصلی کاربر", "callback_data": "U:HOME"}],
    ])

def show_admin_menu(chat_id, user_id):
    send_clean(chat_id, user_id, "👑 <b>پنل مدیریت</b>\\nیکی از گزینه‌ها را انتخاب کنید 👇", admin_menu_kb())

def show_admin_status(chat_id, user_id):
    conn = db()
    cur = conn.cursor()
    cur.execute("SELECT COUNT(*) c FROM users")
    total = cur.fetchone()["c"]
    cur.execute("SELECT COUNT(*) c FROM users WHERE last_active >= ?", ((datetime.now(timezone.utc) - timedelta(days=1)).isoformat(),))
    active = cur.fetchone()["c"]
    conn.close()
    maxb = int(get_setting("max_size_bytes", 0))
    minb = int(get_setting("min_size_bytes", 0))
    text = (
        "📊 <b>وضعیت ربات</b>\\n\\n"
        f"👥 کل کاربران: <b>{total}</b>\\n"
        f"✅ کاربران فعال (24h): <b>{active}</b>\\n"
        f"📦 حداقل حجم: <b>{bytes_human(minb)}</b>\\n"
        f"📦 سقف حجم فعلی: <b>{bytes_human(maxb)}</b>\\n"
        f"🔒 جوین اجباری: <b>{'روشن' if get_setting('join_required', True) else 'خاموش'}</b>\\n"
    )
    send_clean(chat_id, user_id, text, ik([[{"text": "🔄 بروزرسانی", "callback_data": "A:STATUS"}], [{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}]]))

def show_admin_maxsize(chat_id, user_id):
    maxb = int(get_setting("max_size_bytes", 0))
    text = (
        "📦 <b>تنظیم سقف حجم فایل</b>\\n\\n"
        f"سقف فعلی: <b>{bytes_human(maxb)}</b>\\n"
        "مغ مقدار جدید را انتخاب کنید:"
    )
    kb = ik([
        [{"text": "➖ 50MB", "callback_data": "A:MAXSIZE:DEC:50MB"}, {"text": "➕ 50MB", "callback_data": "A:MAXSIZE:INC:50MB"}],
        [{"text": "➖ 200MB", "callback_data": "A:MAXSIZE:DEC:200MB"}, {"text": "➕ 200MB", "callback_data": "A:MAXSIZE:INC:200MB"}],
        [{"text": "✏️ ورود دستی", "callback_data": "A:MAXSIZE:MANUAL"}],
        [{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}],
        [{"text": "❌ لغو", "callback_data": "U:CANCEL"}],
    ])
    send_clean(chat_id, user_id, text, kb)

def show_admin_bcast(chat_id, user_id):
    text = "📝 <b>ارسال همگانی</b>\\n\\nمخاطب را انتخاب کنید:"
    kb = ik([
        [{"text": "👥 همه کاربران", "callback_data": "A:BCAST:ALL"}],
        [{"text": "✅ کاربران فعال 7 روز اخیر", "callback_data": "A:BCAST:ACTIVE7"}],
        [{"text": "🚫 فقط غیرمسدودها", "callback_data": "A:BCAST:UNBLOCKED"}],
        [{"text": "⭐ فقط اعضای جوین اجباری", "callback_data": "A:BCAST:JOINED"}],
        [{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}],
        [{"text": "❌ لغو", "callback_data": "U:CANCEL"}],
    ])
    send_clean(chat_id, user_id, text, kb)

def show_admin_join(chat_id, user_id):
    chans = list_join_channels()
    lines = []
    for i, ch in enumerate(chans, start=1):
        lines.append(f"{i}) <code>{ch['chat_ref']}</code> — {ch.get('title') or '—'}")
    text = "🔒 <b>جوین اجباری</b>\\n\\n"
    text += f"وضعیت: <b>{'روشن' if get_setting('join_required', True) else 'خاموش'}</b>\\n\\n"
    text += "کانال‌ها:\\n" + ("\\n".join(lines) if lines else "— (هیچ کانالی ثبت نشده)")
    kb = ik([
        [{"text": "✅ روشن/خاموش", "callback_data": "A:JOIN:TOGGLE"}],
        [{"text": "➕ افزودن کانال", "callback_data": "A:JOIN:ADD"}],
        [{"text": "🗑 حذف کانال", "callback_data": "A:JOIN:DEL"}],
        [{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}],
    ])
    send_clean(chat_id, user_id, text, kb)

def show_admin_upreq(chat_id, user_id):
    conn = db()
    cur = conn.cursor()
    cur.execute("SELECT id, user_id, requested_plan, created_at FROM upgrade_requests WHERE status='pending' ORDER BY id DESC LIMIT 10")
    rows = cur.fetchall()
    conn.close()
    if not rows:
        send_clean(chat_id, user_id, "📩 <b>درخواست‌های ارتقا</b>\\n\\nهیچ درخواست فعالی نیست.", ik([[{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}]]))
        return
    lines = []
    buttons = []
    for r in rows:
        rid = r["id"]
        uid = r["user_id"]
        rp = r["requested_plan"]
        lines.append(f"• درخواست #{rid} — کاربر <code>{uid}</code> → <b>{'ویژه' if rp=='pro' else rp}</b>")
        buttons.append([{"text": f"✅ تأیید #{rid}", "callback_data": f"A:UPREQ:APP:{rid}"}, {"text": f"❌ رد #{rid}", "callback_data": f"A:UPREQ:REJ:{rid}"}])
    buttons.append([{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}])
    send_clean(chat_id, user_id, "📩 <b>درخواست‌های ارتقا</b>\\n\\n" + "\\n".join(lines), ik(buttons))

# ---------- Flows ----------
def start_rename_file(chat_id, user_id):
    ok, note = check_user_joined_all(user_id)
    if not ok:
        show_join_gate(chat_id, user_id, note)
        return
    send_clean(chat_id, user_id, "📁 لطفاً فایل موردنظر را ارسال کنید.\\n(بهتر است به صورت Document ارسال شود)", ik([[{"text": "❌ لغو", "callback_data": "U:CANCEL"}]]))
    set_user_state(user_id, "WAIT_FILE", {})

def start_rename_link(chat_id, user_id):
    ok, note = check_user_joined_all(user_id)
    if not ok:
        show_join_gate(chat_id, user_id, note)
        return
    send_clean(chat_id, user_id, "🔗 لطفاً <b>لینک مستقیم فایل</b> را ارسال کنید (http/https).", ik([[{"text": "❌ لغو", "callback_data": "U:CANCEL"}]]))
    set_user_state(user_id, "WAIT_URL", {})

def show_file_summary(chat_id, user_id, info, extra=""):
    kind = detect_kind(info.get("mime", ""), info.get("filename", "file"))
    text = f"📄 نام فعلی: <code>{sanitize_filename(info.get('filename','file'))}</code>\\n📦 حجم: <b>{bytes_human(int(info.get('size',0) or 0))}</b>\\n🧩 نوع: <b>{kind}</b>"
    if extra:
        text = extra + "\\n\\n" + text
    send_clean(chat_id, user_id, text + "\\n\\nمی‌خوای چیکار کنم؟", file_action_kb())

def show_confirm(chat_id, user_id, stdata):
    orig = stdata.get("orig_filename") or "file"
    newname = stdata.get("new_filename") or orig
    fmt = stdata.get("format") or ""
    fmt_line = f"🔄 فرمت خروجی: <b>{fmt}</b>\\n" if fmt else ""
    text = f"🔁 <b>تأیید نهایی</b>\\n\\nاز: <code>{sanitize_filename(orig)}</code>\\nبه: <code>{sanitize_filename(newname)}</code>\\n{fmt_line}\\nتأیید می‌کنی؟"
    send_clean(chat_id, user_id, text, confirm_kb())

def enforce_limits_or_fail(chat_id, user_id, file_size):
    u = get_user(user_id)
    if u.get("is_blocked"):
        send_clean(chat_id, user_id, "⛔️ دسترسی شما مسدود است.", ik([[{"text": "🏠 منوی اصلی", "callback_data": "U:HOME"}]]))
        return False
    lim = get_user_limits(u)
    if file_size and file_size < lim["min_size"]:
        send_clean(chat_id, user_id, f"❌ حجم فایل کمتر از حداقل مجاز است.\\nحداقل: {bytes_human(lim['min_size'])}", ik([[{"text": "🏠 منوی اصلی", "callback_data": "U:HOME"}]]))
        return False
    if file_size and file_size > lim["max_size"]:
        send_clean(chat_id, user_id, f"❌ فایل بزرگ‌تر از حد مجاز است.\\nحجم فایل: {bytes_human(file_size)}\\nسقف مجاز: {bytes_human(lim['max_size'])}", ik([[{"text": "🏠 منوی اصلی", "callback_data": "U:HOME"}]]))
        return False
    used = get_daily_usage(user_id)
    if u.get("plan") != "admin" and used >= lim["daily_files"]:
        send_clean(chat_id, user_id, "⚠️ سقف تعداد فایل روزانه شما پر شده است.\\nبرای استفاده بیشتر، ارتقا دهید 👇", ik([[{"text": "🏷 پلن‌ها و ارتقا", "callback_data": "U:PLANS"}], [{"text": "🏠 منوی اصلی", "callback_data": "U:HOME"}]]))
        return False
    return True

def process_and_send(chat_id, user_id, stdata):
    u = get_user(user_id)
    lim = get_user_limits(u)

    src = stdata.get("src")
    orig_filename = stdata.get("orig_filename") or "file"
    new_filename = stdata.get("new_filename") or orig_filename
    fmt = stdata.get("format")  # JPG/PNG/WEBP

    kind = detect_kind(stdata.get("mime", ""), orig_filename)

    with tempfile.TemporaryDirectory() as td:
        td = Path(td)
        in_path = td / "input.bin"
        out_path = td / "output.bin"

        if src == "tg":
            tg_download(stdata["file_id"], in_path)
        elif src == "url":
            download_url_to(stdata["url"], in_path, lim["max_size"])
        else:
            raise RuntimeError("Unknown source")

        final_name = sanitize_filename(new_filename)

        if fmt and kind == "image":
            target_ext = {"JPG": ".jpg", "PNG": ".png", "WEBP": ".webp"}.get(fmt.upper(), "")
            final_name = str(Path(final_name).with_suffix(target_ext))
            img = Image.open(in_path)
            if fmt.upper() == "JPG" and img.mode in ("RGBA", "P"):
                img = img.convert("RGB")
            img.save(out_path, format=fmt.upper())
            send_path = out_path
        else:
            if Path(final_name).suffix == "":
                final_name = str(Path(final_name).with_suffix(Path(orig_filename).suffix))
            send_path = in_path

        with open(send_path, "rb") as f:
            files = {"document": (Path(final_name).name, f)}
            payload = {"chat_id": chat_id, "caption": f"✅ تغییر انجام شد\\nاز: {sanitize_filename(orig_filename)}\\nبه: {sanitize_filename(Path(final_name).name)}"}
            res = tg("sendDocument", payload, files=files, timeout=300)
            if not res.get("ok"):
                send_clean(chat_id, user_id, "❌ ارسال فایل ناموفق بود. لطفاً دوباره تلاش کنید.", ik([[{"text": "🏠 منوی اصلی", "callback_data": "U:HOME"}]]))
                return

    bump_daily_usage(user_id)
    set_user_state(user_id, None, None)
    u = get_user(user_id)
    send_clean(chat_id, user_id, "✅ فایل با موفقیت آماده و ارسال شد.\\nاگر فایل دیگری دارید از منو انتخاب کنید 👇", main_menu_kb(u.get("plan") == "admin"))

# ---------- Webhook endpoints ----------
@app.get("/")
def health():
    return "OK"

@app.post(WEBHOOK_PATH)
def webhook():
    if WEBHOOK_SECRET:
        sec = request.headers.get("X-Telegram-Bot-Api-Secret-Token", "") or request.args.get("secret", "")
        if sec != WEBHOOK_SECRET:
            abort(403)
    upd = request.get_json(force=True, silent=True) or {}
    try:
        handle_update(upd)
    except Exception:
        pass
    return jsonify({"ok": True})

def handle_update(upd):
    if "message" in upd:
        handle_message(upd["message"])
    elif "callback_query" in upd:
        handle_callback(upd["callback_query"])

def handle_message(msg):
    chat = msg.get("chat", {})
    chat_id = chat.get("id")
    from_u = msg.get("from", {})
    if not chat_id or not from_u:
        return
    if chat.get("type") != "private":
        return

    upsert_user(from_u)
    user = get_user(from_u["id"])
    user_id = user["user_id"]

    text = (msg.get("text") or "").strip()

    if text.startswith("/start"):
        ok, note = check_user_joined_all(user_id)
        if not ok:
            show_join_gate(chat_id, user_id, note)
            return
        show_main(chat_id, user_id)
        return

    if text.startswith("/admin"):
        if user.get("plan") != "admin":
            send_clean(chat_id, user_id, "⛔️ دسترسی غیرمجاز", ik([[{"text": "🏠 منوی اصلی", "callback_data": "U:HOME"}]]))
            return
        show_admin_menu(chat_id, user_id)
        return

    state, stdata = get_state(user_id)

    # Admin manual input (max size / join add-del / broadcast / join msg)
    if user.get("plan") == "admin":
        if state == "A_WAIT_MAXSIZE":
            b = parse_size(text)
            if b is None:
                send_clean(chat_id, user_id, "❌ فرمت حجم نامعتبر است. مثال: 500MB یا 2GB", ik([[{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}]]))
                set_user_state(user_id, None, None)
                return
            minb = int(get_setting("min_size_bytes", 1_048_576))
            max_allowed = 3 * 1024**3
            if b < minb or b > max_allowed:
                send_clean(chat_id, user_id, f"❌ مقدار باید بین {bytes_human(minb)} تا {bytes_human(max_allowed)} باشد.", ik([[{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}]]))
                set_user_state(user_id, None, None)
                return
            set_setting("max_size_bytes", int(b))
            log_admin(user_id, "set_max_size", str(b))
            show_admin_maxsize(chat_id, user_id)
            set_user_state(user_id, None, None)
            return

        if state == "A_WAIT_JOIN_ADD":
            ref = text.strip()
            if not (ref.startswith("@") or ref.startswith("-100") or ref.lstrip("-").isdigit()):
                send_clean(chat_id, user_id, "❌ لطفاً @username یا chat_id (مثل -100...) را وارد کنید.", ik([[{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}]]))
                set_user_state(user_id, None, None)
                return
            res = tg_get("getChat", {"chat_id": ref}, timeout=20)
            title = res["result"].get("title") if res.get("ok") else None
            conn = db()
            cur = conn.cursor()
            cur.execute("INSERT INTO join_channels(chat_ref, title, invite_link, created_at) VALUES(?,?,?,?)", (ref, title, None, now_iso()))
            conn.commit()
            conn.close()
            log_admin(user_id, "add_join_channel", ref)
            show_admin_join(chat_id, user_id)
            set_user_state(user_id, None, None)
            return

        if state == "A_WAIT_JOIN_DEL":
            try:
                idx = int(text.strip())
            except Exception:
                send_clean(chat_id, user_id, "❌ یک شماره وارد کنید (مثلاً 1).", ik([[{"text": "🏠 برگشت", "callback_data": "A:JOIN"}]]))
                return
            chans = list_join_channels()
            if idx < 1 or idx > len(chans):
                send_clean(chat_id, user_id, "❌ شماره نامعتبر است.", ik([[{"text": "🏠 برگشت", "callback_data": "A:JOIN"}]]))
                return
            ref = chans[idx - 1]["chat_ref"]
            conn = db()
            cur = conn.cursor()
            cur.execute("DELETE FROM join_channels WHERE chat_ref=?", (ref,))
            conn.commit()
            conn.close()
            log_admin(user_id, "del_join_channel", ref)
            show_admin_join(chat_id, user_id)
            set_user_state(user_id, None, None)
            return

        if state == "A_WAIT_BCAST_CONTENT":
            stdata["content"] = {"text": text}
            set_user_state(user_id, "A_WAIT_BCAST_CONFIRM", stdata)
            send_clean(chat_id, user_id, "✅ پیام دریافت شد. ارسال را شروع کنم؟", ik([
                [{"text": "✅ ارسال کن", "callback_data": "A:BCAST:RUN"}],
                [{"text": "✏️ دوباره بفرست", "callback_data": "A:BCAST:EDIT"}],
                [{"text": "❌ لغو", "callback_data": "U:CANCEL"}],
            ]))
            return

    # User states
    if state == "WAIT_URL":
        url = text
        try:
            p = urllib.parse.urlparse(url)
            if p.scheme not in ("http", "https"):
                raise ValueError("bad")
        except Exception:
            send_clean(chat_id, user_id, "❌ لینک نامعتبر است. فقط http/https.", ik([[{"text": "❌ لغو", "callback_data": "U:CANCEL"}]]))
            return
        guess = sanitize_filename(guess_filename_from_url(url))
        st = {"src": "url", "url": url, "orig_filename": guess, "mime": "", "size": 0}
        set_user_state(user_id, "GOT_FILE", st)
        show_file_summary(chat_id, user_id, {"filename": guess, "mime": "", "size": 0}, extra="✅ لینک دریافت شد.")
        return

    if state == "WAIT_FILE":
        finfo = get_file_info_from_message(msg)
        if not finfo:
            send_clean(chat_id, user_id, "❌ لطفاً فایل ارسال کنید (Document/Photo/Video/Audio).", ik([[{"text": "❌ لغو", "callback_data": "U:CANCEL"}]]))
            return
        size = int(finfo.get("size") or 0)
        if size and not enforce_limits_or_fail(chat_id, user_id, size):
            set_user_state(user_id, None, None)
            return
        st = {"src": "tg", "file_id": finfo["file_id"], "orig_filename": finfo["filename"], "mime": finfo.get("mime") or "", "size": size}
        set_user_state(user_id, "GOT_FILE", st)
        show_file_summary(chat_id, user_id, finfo, extra="✅ فایل دریافت شد.")
        return

    if state == "WAIT_NEWNAME":
        newraw = sanitize_filename(text)
        orig = stdata.get("orig_filename") or "file"
        ext = Path(orig).suffix
        newname = newraw + ext if Path(newraw).suffix == "" else newraw
        stdata["new_filename"] = newname
        set_user_state(user_id, "GOT_FILE", stdata)
        show_confirm(chat_id, user_id, stdata)
        return

def run_broadcast(target, content):
    text = content.get("text", "")
    conn = db()
    cur = conn.cursor()
    q = "SELECT user_id FROM users"
    params = ()
    if target == "ACTIVE7":
        q = "SELECT user_id FROM users WHERE last_active >= ?"
        params = ((datetime.now(timezone.utc) - timedelta(days=7)).isoformat(),)
    elif target == "UNBLOCKED":
        q = "SELECT user_id FROM users WHERE is_blocked=0"
    cur.execute(q, params)
    ids = [r["user_id"] for r in cur.fetchall()]
    conn.close()

    okc = 0
    failc = 0
    for uid in ids:
        try:
            if target == "JOINED":
                ok, _ = check_user_joined_all(uid)
                if not ok:
                    continue
            r = tg("sendMessage", {"chat_id": uid, "text": text, "parse_mode": "HTML"}, timeout=20)
            okc += 1 if r.get("ok") else 0
            failc += 0 if r.get("ok") else 1
            time.sleep(0.04)
        except Exception:
            failc += 1
    return okc, failc

def handle_upgrade_decision(chat_id, admin_id, rid, approve, message_id=None):
    if message_id:
        safe_delete(chat_id, message_id)
    conn = db()
    cur = conn.cursor()
    cur.execute("SELECT user_id, requested_plan FROM upgrade_requests WHERE id=? AND status='pending'", (rid,))
    row = cur.fetchone()
    if not row:
        conn.close()
        send_clean(chat_id, admin_id, "ℹ️ درخواست پیدا نشد یا قبلاً بررسی شده.", ik([[{"text": "📩 درخواست‌ها", "callback_data": "A:UPREQ"}], [{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}]]))
        return
    uid = row["user_id"]
    if approve:
        cur.execute("UPDATE users SET plan='pro' WHERE user_id=?", (uid,))
        status = "approved"
    else:
        status = "rejected"
    cur.execute("UPDATE upgrade_requests SET status=?, decided_at=?, decided_by=? WHERE id=?", (status, now_iso(), admin_id, rid))
    conn.commit()
    conn.close()
    log_admin(admin_id, "upgrade_decision", f"rid={rid} {status}")

    try:
        if approve:
            tg("sendMessage", {"chat_id": uid, "text": "✅ درخواست ارتقای شما تأیید شد. پلن شما به «ویژه» تغییر کرد.", "parse_mode": "HTML"})
        else:
            tg("sendMessage", {"chat_id": uid, "text": "❌ درخواست ارتقای شما رد شد.", "parse_mode": "HTML"})
    except Exception:
        pass
    show_admin_upreq(chat_id, admin_id)

def handle_callback(cb):
    data = cb.get("data") or ""
    from_u = cb.get("from") or {}
    msg = cb.get("message") or {}
    chat_id = msg.get("chat", {}).get("id")
    message_id = msg.get("message_id")
    if not chat_id or not from_u:
        return

    upsert_user(from_u)
    user = get_user(from_u["id"])
    user_id = user["user_id"]

    try:
        tg("answerCallbackQuery", {"callback_query_id": cb.get("id")})
    except Exception:
        pass

    if data == "U:CANCEL":
        if message_id:
            safe_delete(chat_id, message_id)
        set_user_state(user_id, None, None)
        send_clean(chat_id, user_id, get_setting("msg_cancelled", "عملیات لغو شد ❌"), cancel_menu_kb(user.get("plan") == "admin"))
        return

    if data == "U:HOME":
        if message_id:
            safe_delete(chat_id, message_id)
        ok, note = check_user_joined_all(user_id)
        if not ok:
            show_join_gate(chat_id, user_id, note)
            return
        show_main(chat_id, user_id)
        return

    if data == "U:HELP":
        if message_id:
            safe_delete(chat_id, message_id)
        show_help(chat_id, user_id)
        return

    if data == "U:PROFILE":
        if message_id:
            safe_delete(chat_id, message_id)
        ok, note = check_user_joined_all(user_id)
        if not ok:
            show_join_gate(chat_id, user_id, note)
            return
        show_profile(chat_id, user_id)
        return

    if data == "U:PLANS":
        if message_id:
            safe_delete(chat_id, message_id)
        show_plans(chat_id, user_id)
        return

    if data.startswith("U:REQ_UP:"):
        req_plan = data.split(":")[-1]
        if user.get("plan") == "admin":
            return
        ok = create_upgrade_request(user_id, req_plan)
        if message_id:
            safe_delete(chat_id, message_id)
        if ok:
            send_clean(chat_id, user_id, "✅ درخواست ارتقای شما ثبت شد. به محض بررسی اطلاع می‌دهیم.", ik([[{"text": "👤 پروفایل من", "callback_data": "U:PROFILE"}], [{"text": "🏠 منوی اصلی", "callback_data": "U:HOME"}]]))
            notify_admins(f"📩 درخواست ارتقا\\nکاربر <code>{user_id}</code> درخواست ارتقا به <b>ویژه</b> ثبت کرد.")
        else:
            send_clean(chat_id, user_id, "ℹ️ یک درخواست ارتقا در حال بررسی دارید.", ik([[{"text": "👤 پروفایل من", "callback_data": "U:PROFILE"}], [{"text": "🏠 منوی اصلی", "callback_data": "U:HOME"}]]))
        return

    if data == "J:CHECK":
        if message_id:
            safe_delete(chat_id, message_id)
        ok, note = check_user_joined_all(user_id)
        if not ok:
            show_join_gate(chat_id, user_id, note)
            return
        show_main(chat_id, user_id)
        return

    if data == "U:RENAME_FILE":
        if message_id:
            safe_delete(chat_id, message_id)
        start_rename_file(chat_id, user_id)
        return

    if data == "U:RENAME_LINK":
        if message_id:
            safe_delete(chat_id, message_id)
        start_rename_link(chat_id, user_id)
        return

    state, stdata = get_state(user_id)

    if data == "U:SET_NAME":
        if message_id:
            safe_delete(chat_id, message_id)
        ext = Path(stdata.get("orig_filename", "file")).suffix
        send_clean(chat_id, user_id, f"✏️ اسم جدید فایل را ارسال کنید.\\nاگر پسوند ندهید، پسوند <b>{ext or '—'}</b> حفظ می‌شود.", ik([[{"text": "❌ لغو", "callback_data": "U:CANCEL"}]]))
        set_user_state(user_id, "WAIT_NEWNAME", stdata)
        return

    if data == "U:SET_FMT":
        if message_id:
            safe_delete(chat_id, message_id)
        kind = detect_kind(stdata.get("mime", ""), stdata.get("orig_filename", "file"))
        send_clean(chat_id, user_id, "🔄 فرمت خروجی را انتخاب کنید:", format_kb_for(kind))
        set_user_state(user_id, "WAIT_FORMAT", stdata)
        return

    if data.startswith("F:"):
        fmt = data.split(":")[1]
        stdata["format"] = fmt
        if message_id:
            safe_delete(chat_id, message_id)
        set_user_state(user_id, "GOT_FILE", stdata)
        show_confirm(chat_id, user_id, stdata)
        return

    if data == "U:BACK_FILE":
        if message_id:
            safe_delete(chat_id, message_id)
        info = {"filename": stdata.get("orig_filename", "file"), "mime": stdata.get("mime", ""), "size": stdata.get("size", 0)}
        set_user_state(user_id, "GOT_FILE", stdata)
        show_file_summary(chat_id, user_id, info)
        return

    if data == "U:SEND":
        if message_id:
            safe_delete(chat_id, message_id)
        size = int(stdata.get("size") or 0)
        if size and not enforce_limits_or_fail(chat_id, user_id, size):
            set_user_state(user_id, None, None)
            return
        send_clean(chat_id, user_id, "⏳ در حال آماده‌سازی فایل…", None)
        try:
            process_and_send(chat_id, user_id, stdata)
        except Exception as e:
            send_clean(chat_id, user_id, f"❌ خطا در پردازش: {e}", ik([[{"text": "🏠 منوی اصلی", "callback_data": "U:HOME"}]]))
            set_user_state(user_id, None, None)
        return

    # Admin callbacks
    if data.startswith("A:"):
        if user.get("plan") != "admin":
            send_clean(chat_id, user_id, "⛔️ دسترسی غیرمجاز", ik([[{"text": "🏠 منوی اصلی", "callback_data": "U:HOME"}]]))
            return

        if data == "A:MENU":
            if message_id:
                safe_delete(chat_id, message_id)
            show_admin_menu(chat_id, user_id)
            return

        if data == "A:STATUS":
            if message_id:
                safe_delete(chat_id, message_id)
            show_admin_status(chat_id, user_id)
            return

        if data == "A:MAXSIZE":
            if message_id:
                safe_delete(chat_id, message_id)
            show_admin_maxsize(chat_id, user_id)
            return

        if data == "A:MAXSIZE:MANUAL":
            if message_id:
                safe_delete(chat_id, message_id)
            send_clean(chat_id, user_id, "✏️ حجم جدید را وارد کنید (مثال: 500MB یا 2GB)\\n(حداکثر 3GB، حداقل 1MB)", ik([[{"text": "❌ لغو", "callback_data": "U:CANCEL"}], [{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}]]))
            set_user_state(user_id, "A_WAIT_MAXSIZE", {})
            return

        if data.startswith("A:MAXSIZE:") and data.count(":") == 3:
            _, _, op, step = data.split(":")
            stepb = parse_size(step) or 0
            cur = int(get_setting("max_size_bytes", 0))
            minb = int(get_setting("min_size_bytes", 1_048_576))
            max_allowed = 3 * 1024**3
            cur = cur + stepb if op == "INC" else cur - stepb
            cur = max(minb, min(cur, max_allowed))
            set_setting("max_size_bytes", int(cur))
            log_admin(user_id, "adjust_max_size", f"{op} {step}")
            if message_id:
                safe_delete(chat_id, message_id)
            show_admin_maxsize(chat_id, user_id)
            return

        if data == "A:BCAST":
            if message_id:
                safe_delete(chat_id, message_id)
            show_admin_bcast(chat_id, user_id)
            return

        if data.startswith("A:BCAST:") and data.count(":") == 2:
            target = data.split(":")[2]
            if message_id:
                safe_delete(chat_id, message_id)
            send_clean(chat_id, user_id, "📝 لطفاً متن پیام همگانی را ارسال کنید.", ik([[{"text": "❌ لغو", "callback_data": "U:CANCEL"}], [{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}]]))
            set_user_state(user_id, "A_WAIT_BCAST_CONTENT", {"target": target})
            return

        if data == "A:BCAST:EDIT":
            if message_id:
                safe_delete(chat_id, message_id)
            st, stdata2 = get_state(user_id)
            target = stdata2.get("target", "ALL")
            send_clean(chat_id, user_id, "📝 لطفاً متن جدید را ارسال کنید.", ik([[{"text": "❌ لغو", "callback_data": "U:CANCEL"}], [{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}]]))
            set_user_state(user_id, "A_WAIT_BCAST_CONTENT", {"target": target})
            return

        if data == "A:BCAST:RUN":
            st, stdata2 = get_state(user_id)
            content = stdata2.get("content", {})
            target = stdata2.get("target", "ALL")
            if message_id:
                safe_delete(chat_id, message_id)
            send_clean(chat_id, user_id, "⏳ ارسال همگانی شروع شد…", None)
            okc, failc = run_broadcast(target, content)
            log_admin(user_id, "broadcast", f"{target} ok={okc} fail={failc}")
            send_clean(chat_id, user_id, f"✅ ارسال تمام شد.\\nموفق: {okc}\\nناموفق: {failc}", ik([[{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}]]))
            set_user_state(user_id, None, None)
            return

        if data == "A:JOIN":
            if message_id:
                safe_delete(chat_id, message_id)
            show_admin_join(chat_id, user_id)
            return

        if data == "A:JOIN:TOGGLE":
            cur = bool(get_setting("join_required", True))
            set_setting("join_required", not cur)
            log_admin(user_id, "toggle_join_required", str(not cur))
            if message_id:
                safe_delete(chat_id, message_id)
            show_admin_join(chat_id, user_id)
            return

        if data == "A:JOIN:ADD":
            if message_id:
                safe_delete(chat_id, message_id)
            send_clean(chat_id, user_id, "➕ @username کانال/گروه یا chat_id را ارسال کنید.\\n(برای کانال خصوصی بهتر است chat_id مثل -100... باشد)", ik([[{"text": "❌ لغو", "callback_data": "U:CANCEL"}], [{"text": "🏠 منوی ادمین", "callback_data": "A:MENU"}]]))
            set_user_state(user_id, "A_WAIT_JOIN_ADD", {})
            return

        if data == "A:JOIN:DEL":
            if message_id:
                safe_delete(chat_id, message_id)
            send_clean(chat_id, user_id, "🗑 شماره کانال برای حذف را ارسال کنید (از لیست). مثال: 1", ik([[{"text": "🏠 برگشت", "callback_data": "A:JOIN"}], [{"text": "❌ لغو", "callback_data": "U:CANCEL"}]]))
            set_user_state(user_id, "A_WAIT_JOIN_DEL", {})
            return

        if data == "A:UPREQ":
            if message_id:
                safe_delete(chat_id, message_id)
            show_admin_upreq(chat_id, user_id)
            return

        if data.startswith("A:UPREQ:APP:"):
            rid = int(data.split(":")[-1])
            handle_upgrade_decision(chat_id, user_id, rid, True, message_id=message_id)
            return

        if data.startswith("A:UPREQ:REJ:"):
            rid = int(data.split(":")[-1])
            handle_upgrade_decision(chat_id, user_id, rid, False, message_id=message_id)
            return

# Register callback dispatcher
def handle_callback_dispatch(cb):
    handle_callback(cb)

# Ensure update dispatcher uses correct callback
def handle_update(upd):
    if "message" in upd:
        handle_message(upd["message"])
    elif "callback_query" in upd:
        handle_callback(upd["callback_query"])

# ---------- Init on import ----------
init_db()
ensure_defaults()
